import { h, Host } from '@stencil/core';
const DATE_TYPE_FORMATS = {
  t: { timeStyle: 'short' },
  T: { timeStyle: 'medium' },
  d: { dateStyle: 'short' },
  D: { dateStyle: 'long' },
  f: { dateStyle: 'long', timeStyle: 'short' },
  F: { dateStyle: 'full', timeStyle: 'short' },
  R: { style: 'long', numeric: 'auto' }
};
// max: [unit, per unit]
const RELATIVE_DATE_CONVERSION = {
  60000: ['second', 1000],
  3600000: ['minute', 60000],
  86400000: ['hour', 3600000],
  604800000: ['day', 86400000],
  2419200000: ['week', 604800000],
  29030400000: ['month', 2419200000],
  290304000000: ['year', 29030400000]
};
export class DiscordTime {
  constructor() {
    this.timestamp = undefined;
    this.format = 't';
    this.time = '';
  }
  render() {
    return h(Host, { class: "discord-time" }, this.time);
  }
  // Lifecycle methods
  connectedCallback() {
    this.update();
  }
  disconnectedCallback() {
    window.clearInterval(this.updateInterval);
  }
  /**
   * Generates a string for the time.
   */
  update() {
    const date = new Date(this.timestamp);
    if (this.format === 'R') {
      const [formatted, interval] = getRelativeDate(date);
      this.time = formatted;
      // Update the time according to the interval
      if (this.updateInterval)
        window.clearInterval(this.updateInterval);
      if (interval > -1)
        this.updateInterval = window.setInterval(() => this.update(), interval);
    }
    else {
      this.time = date.toLocaleString(undefined, DATE_TYPE_FORMATS[this.format]);
    }
  }
  static get is() { return "discord-time"; }
  static get originalStyleUrls() {
    return {
      "$": ["discord-time.css"]
    };
  }
  static get styleUrls() {
    return {
      "$": ["discord-time.css"]
    };
  }
  static get properties() {
    return {
      "timestamp": {
        "type": "number",
        "mutable": false,
        "complexType": {
          "original": "number",
          "resolved": "number",
          "references": {}
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [],
          "text": "The time to display."
        },
        "attribute": "timestamp",
        "reflect": false
      },
      "format": {
        "type": "string",
        "mutable": false,
        "complexType": {
          "original": "'t' | 'T' | 'f' | 'F' | 'd' | 'D' | 'R'",
          "resolved": "\"D\" | \"F\" | \"R\" | \"T\" | \"d\" | \"f\" | \"t\"",
          "references": {}
        },
        "required": false,
        "optional": false,
        "docs": {
          "tags": [],
          "text": "The format for the time."
        },
        "attribute": "format",
        "reflect": false,
        "defaultValue": "'t'"
      }
    };
  }
  static get states() {
    return {
      "time": {}
    };
  }
}
// [formatted, updateInterval]
function getRelativeDate(date) {
  const difference = Date.now() - date.getTime();
  const diffAbsolute = Math.abs(difference);
  const ending = difference < 0 ? 'from now' : 'ago';
  if (diffAbsolute < 5000) {
    return ['Just now', 1000];
  }
  for (const [time, [unit, per]] of Object.entries(RELATIVE_DATE_CONVERSION)) {
    if (diffAbsolute < Number(time)) {
      const amount = Math.round(diffAbsolute / per);
      return [`${amount} ${unit}${amount === 1 ? '' : 's'} ${ending}`, unit === 'second' ? 1000 : 60 * 1000];
    }
  }
  return [`${Math.floor(diffAbsolute / 290304000000)} years ${ending}`, -1];
}
//# sourceMappingURL=discord-time.js.map
